<?php
/**
 * PluginInventory - Erstellt ein Inventar aller Plugins
 * 
 * Statt alle Plugins zu sichern, wird nur eine Liste erstellt:
 * - WP.org Plugins: Werden beim Restore automatisch installiert
 * - Premium Plugins: Werden gesichert
 * - Custom Plugins: Werden gesichert
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Backup;

if (!defined('ABSPATH')) {
    exit;
}

class PluginInventory {
    
    /** @var array Bekannte Premium-Plugin-Patterns */
    private $premium_patterns = [
        'elementor-pro',
        'advanced-custom-fields-pro',
        'acf-pro',
        'gravityforms',
        'wpforms-pro',
        'ninja-forms-pro',
        'wp-rocket',
        'updraftplus-premium',
        'bricks', // Bricks Builder
        'oxygen',
        'divi',
        'astra-pro',
        'generatepress-premium',
        'rank-math-pro',
        'yoast-seo-premium',
        'woocommerce-subscriptions',
        'woocommerce-memberships',
        'woocommerce-bookings',
        'learndash',
        'memberpress',
        'wpml',
        'polylang-pro',
    ];
    
    /** @var array WP.org API Cache */
    private $wporg_cache = [];
    
    /**
     * Sammelt das Plugin-Inventar
     * 
     * @return array Plugin-Liste
     */
    public function collect(): array {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', []);
        $inventory = [];
        
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $slug = dirname($plugin_file);
            if ($slug === '.') {
                $slug = basename($plugin_file, '.php');
            }
            
            $source = $this->determineSource($slug, $plugin_file, $plugin_data);
            
            $inventory[] = [
                'slug' => $slug,
                'file' => $plugin_file,
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'author' => $plugin_data['Author'] ?? '',
                'active' => in_array($plugin_file, $active_plugins),
                'source' => $source,
                'update_url' => $plugin_data['UpdateURI'] ?? null,
                'backed_up' => $source !== 'wordpress.org', // Nur Non-WP.org sichern
                'wporg_url' => $source === 'wordpress.org' ? "https://wordpress.org/plugins/{$slug}/" : null,
            ];
        }
        
        return $inventory;
    }
    
    /**
     * Bestimmt die Quelle eines Plugins
     * 
     * @param string $slug Plugin-Slug
     * @param string $plugin_file Plugin-Datei
     * @param array $plugin_data Plugin-Daten
     * @return string 'wordpress.org', 'premium', 'custom'
     */
    private function determineSource(string $slug, string $plugin_file, array $plugin_data): string {
        // 1. Bekannte Premium-Patterns prüfen
        foreach ($this->premium_patterns as $pattern) {
            if (stripos($slug, $pattern) !== false) {
                return 'premium';
            }
        }
        
        // 2. UpdateURI prüfen (Plugin Header)
        if (!empty($plugin_data['UpdateURI'])) {
            if (strpos($plugin_data['UpdateURI'], 'wordpress.org') !== false) {
                return 'wordpress.org';
            }
            return 'premium';
        }
        
        // 3. WP.org API prüfen
        if ($this->isOnWordPressOrg($slug)) {
            return 'wordpress.org';
        }
        
        // 4. Plugin-URI prüfen
        $plugin_uri = $plugin_data['PluginURI'] ?? '';
        if (strpos($plugin_uri, 'wordpress.org') !== false) {
            return 'wordpress.org';
        }
        
        // 5. Fallback: Custom
        return 'custom';
    }
    
    /**
     * Prüft ob Plugin auf WordPress.org verfügbar ist
     * 
     * @param string $slug Plugin-Slug
     * @return bool
     */
    private function isOnWordPressOrg(string $slug): bool {
        // Cache prüfen
        if (isset($this->wporg_cache[$slug])) {
            return $this->wporg_cache[$slug];
        }
        
        // API anfragen
        $url = "https://api.wordpress.org/plugins/info/1.2/?action=plugin_information&slug=" . urlencode($slug);
        
        $response = wp_remote_get($url, [
            'timeout' => 5,
            'sslverify' => false,
        ]);
        
        if (is_wp_error($response)) {
            $this->wporg_cache[$slug] = false;
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // Wenn Daten vorhanden und kein Fehler
        $is_wporg = !empty($data) && !isset($data['error']);
        
        $this->wporg_cache[$slug] = $is_wporg;
        
        return $is_wporg;
    }
    
    /**
     * Gibt Plugins zurück, die gesichert werden müssen
     * 
     * @param array $inventory Plugin-Inventar
     * @return array Zu sichernde Plugins
     */
    public function getPluginsToBackup(array $inventory): array {
        return array_filter($inventory, function($plugin) {
            return $plugin['backed_up'] === true;
        });
    }
    
    /**
     * Gibt WP.org Plugins zurück
     * 
     * @param array $inventory Plugin-Inventar
     * @return array WP.org Plugins
     */
    public function getWpOrgPlugins(array $inventory): array {
        return array_filter($inventory, function($plugin) {
            return $plugin['source'] === 'wordpress.org';
        });
    }
    
    /**
     * Erstellt einen Installations-Plan für den Restore
     * 
     * @param array $inventory Plugin-Inventar
     * @return array Installations-Plan
     */
    public function createInstallPlan(array $inventory): array {
        $plan = [
            'auto_install' => [], // WP.org Plugins
            'manual_restore' => [], // Premium/Custom aus Backup
            'skip' => [], // Bereits installiert
        ];
        
        $installed = array_keys(get_plugins());
        
        foreach ($inventory as $plugin) {
            $plugin_file = $plugin['file'];
            
            // Bereits installiert?
            if (in_array($plugin_file, $installed)) {
                $plan['skip'][] = $plugin;
                continue;
            }
            
            if ($plugin['source'] === 'wordpress.org') {
                $plan['auto_install'][] = $plugin;
            } else {
                $plan['manual_restore'][] = $plugin;
            }
        }
        
        return $plan;
    }
    
    /**
     * Lädt Plugin-Info von WP.org
     * 
     * @param string $slug Plugin-Slug
     * @return array|null Plugin-Info oder null
     */
    public function getWpOrgPluginInfo(string $slug): ?array {
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        
        $api = plugins_api('plugin_information', [
            'slug' => $slug,
            'fields' => [
                'short_description' => true,
                'sections' => false,
                'versions' => true,
                'download_link' => true,
            ],
        ]);
        
        if (is_wp_error($api)) {
            return null;
        }
        
        return (array) $api;
    }
    
    /**
     * Speichert das Inventar als JSON
     * 
     * @param array $inventory Plugin-Inventar
     * @param string $path Zielpfad
     * @return bool
     */
    public function saveInventory(array $inventory, string $path): bool {
        $json = json_encode([
            'generated_at' => gmdate('Y-m-d\TH:i:s\Z'),
            'wordpress_version' => get_bloginfo('version'),
            'plugins' => $inventory,
            'summary' => [
                'total' => count($inventory),
                'active' => count(array_filter($inventory, fn($p) => $p['active'])),
                'wporg' => count(array_filter($inventory, fn($p) => $p['source'] === 'wordpress.org')),
                'premium' => count(array_filter($inventory, fn($p) => $p['source'] === 'premium')),
                'custom' => count(array_filter($inventory, fn($p) => $p['source'] === 'custom')),
            ],
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        
        return file_put_contents($path, $json) !== false;
    }
    
    /**
     * Lädt ein Inventar aus JSON
     * 
     * @param string $path Pfad zur JSON-Datei
     * @return array
     */
    public static function loadInventory(string $path): array {
        if (!file_exists($path)) {
            throw new \Exception("Plugin-Inventar nicht gefunden: $path");
        }
        
        $json = file_get_contents($path);
        $data = json_decode($json, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception("Ungültiges Plugin-Inventar: " . json_last_error_msg());
        }
        
        return $data['plugins'] ?? [];
    }
}

